package org.silex.pdf2swf;
import org.silex.publication.PageModel;
import org.silex.publication.ComponentModel;
import org.silex.serverApi.FileSystemTools;
using StringTools;
#if neko
import neko.io.Process;
import neko.io.File;
#end
#if php
import php.io.Process;
import php.io.File;
#end

class PDF2SWFConverter
{
	//Consts for names used by templates
	private static inline var NORMAL_PAGES = "template_import_pdf.xml";
	private static inline var FIRST_PAGE = "template_import_pdf_first_page.xml";
	private static inline var LAST_PAGE = "template_import_pdf_last_page.xml";
	private static inline var LEFT_COMPONENT = "template_left_page_pdf";
	private static inline var RIGHT_COMPONENT = "template_right_page_pdf";

	/**
	*  This function convert a PDF page into the SWF format.<br/>
	*  Traces "Done" if everything went well.<br/>
	*  Traces "PageDoesNotExist" if the requested page does not exist.<br/>
	*  Traces "Error" in case an other error happened.
	*/
	public static function generatePageSWF(pathToFile : String, options : ConverterOptions, pageNumber : Int) : ReturnCode
	{
		createDirPath(options.swfDirectoryName);
		untyped __call__("set_include_path", __call__("get_include_path") + ":" + php.Sys.getCwd() + "cgi/library");
		untyped __call__("require_once", php.Sys.getCwd() + "cgi/includes/file_system_tools.php");
		untyped __call__("require_once", php.Sys.getCwd() + "cgi/includes/server_config.php");
		
		var serverConfig : Dynamic = untyped __call__("new server_config");
		var contentFolder : String;
		contentFolder = untyped serverConfig.getContentFolderForPublication(options.publicationName);

		var workingDirectory : String = untyped __call__("getcwd");
		untyped __call__("session_start");
		var cl = new FileSystemTools();

		debug("New run.");
		debug("Working directory is " + untyped __call__("getcwd"));
		
		//Sanitize and check rights
		var swfDirectoryName:String = cl.sanitize( options.swfDirectoryName , true );
		if(!cl.checkRights( swfDirectoryName, FileSystemTools.adminRole, FileSystemTools.writeAction ) )
			return ReturnCode.Error("Not enough rights to generate page");

		//Run the pdf2swf process on the page
			
		var proc = new Process(concatPath(options.pathToSWFTools, "pdf2swf"), [concatPath(workingDirectory ,pathToFile), "-s","poly2bitmap", "-s", "zoom=100", "-j 60", "-p", Std.string(pageNumber), "-o", concatPath(workingDirectory ,options.swfDirectoryName + "/" + options.namePrefix + "%.swf")]);
		//trace([pathToFile, "-s","poly2bitmap", "-s", "zoom=100", "-j 60", "-p", Std.string(pageNumber), "-o", options.swfDirectoryName + "/" + options.namePrefix + "-%.swf"]);
		var pagePDF : Int = null;
		var line="";
		var nbLines = 0;
		try
		{
			while(true)
			{
				line = proc.stdout.readLine();
				debug(line);
				nbLines++;
				//If we are processing a new page
				if(line.indexOf("NOTICE  processing PDF page ") == 0)
				{
					pagePDF = Std.parseInt(line.split(" ")[5]);
				}
			}
		} catch(e : Dynamic) //This happens when the process ends
		{
			if(pagePDF == null && nbLines ==0) //Didn't try to parse any page? => SWFTools certainly not available || PageDoesNotExist
			{
				return ReturnCode.Error("PagePDF was null - Make sure SWFTools are available.");
				//php.Lib.print("Error");
			} else if(pagePDF == null) //No page has been parsed
			{
				return ReturnCode.PageDoesNotExist;
			}
		}
		if(proc.exitCode() == 0 && pagePDF==null)
		{
			//Page doesn't exist
			return ReturnCode.PageDoesNotExist;
		}
		if(proc.exitCode() != 0)
		{
			//There was an unknown error
			return ReturnCode.Error("Bad exitcode");
		}
		

		//Generate JPEG equivalent
		new Process(concatPath(options.pathToSWFTools,"swfrender"), [concatPath(workingDirectory, options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".swf"), "-o", concatPath(workingDirectory, options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".png")]);
		var imgpng = untyped __call__("imagecreatefrompng", options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".png");
		var size = php.Lib.toHaxeArray(untyped __call__("getimagesize", options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".png"));
		var imgjpg = untyped __call__("imagecreatetruecolor", size[0], size[1]);
		untyped __call__("imagecopy", imgjpg, imgpng, 0, 0, 0, 0, size[0], size[1]);
		untyped __call__("imagejpeg",imgjpg, options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".jpg", 75); 
		//Delete unneeded PNG files
		php.FileSystem.deleteFile(options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".png");
		return ReturnCode.Done(options.swfDirectoryName + "/" + options.namePrefix + pageNumber + ".jpg");
	}
	

	/**
	*  Creates or alters the PublicationStructure to include PDF pages equivalents (will include SWF files).<br/>
	*  @arg options Converter's options<br/>
	*  @arg lastPage The number of the last PDF page to include<br/>
	*  @arg startPage The number of the first PDF page to include<br/>
	*  @arg silexPageNumber The number of the first org.silex.publication.Page in which to include a PDF page
	*/
	public static function generatePublication(options : ConverterOptions, lastPage : Int, startPage : Int, silexPageNumber : Int, layerName : String, importFormat : String)
	{
		var previewsURL : List<String> = new List<String>();
		var templateType : TemplateType;
		
		untyped __call__("session_start");

		untyped __call__("set_include_path", __call__("get_include_path") + ":" + php.Sys.getCwd() + "cgi/library");
		untyped __call__("require_once", php.Sys.getCwd() + "cgi/includes/server_config.php");
		untyped __call__("require_once", php.Sys.getCwd() + "cgi/includes/file_system_tools.php");
		
		var serverConfig : Dynamic = untyped __call__("new server_config");
		var cl = new FileSystemTools();
		var contentFolder : String = untyped serverConfig.getContentFolderForPublication(options.publicationName);
		var contentFolderSanitized : String = cl.sanitize( contentFolder , true );

		//Sanitize and check rights
		if(!cl.checkRights( contentFolderSanitized , FileSystemTools.adminRole, FileSystemTools.writeAction ) )
			return ReturnCode.Error("Not enough rights to generate publication");
		
		var importExtension = switch(importFormat)
								{
									case "SWF":
										".swf";
									default:
										".jpg";
								};
								
		//Those are used when generating preview snapshots
		var snapTop = 0;
		var snapLeft = 0;
		var snapWidth = 500;
		var snapHeight = 500;
		
		//Creates the PublicationStructure
		var ps : org.silex.publication.PublicationStructureModel;
		ps = new org.silex.publication.PublicationStructureModel();
		//trace("structure created");
		
		//Set hasDoublePages & hasCover and recalculate page numbers
		ps.hasCover = options.hasCover;
		ps.hasDoublePages = options.hasDoublePages;
		ps.recalculatePagesNumbers();
		
		//Links between PDF pages number and silex pages number
		//Keys are PDF pages number
		var pagesLinks : IntHash<Int>;
		pagesLinks = new IntHash<Int>();
		for(pn in startPage...lastPage+1)
		{
				pagesLinks.set(pn, Std.int(silexPageNumber) + Std.int(pn));
		}
//		trace(pagesLinks);
		//Create PageModel before silexPageNumber if needed
		for(i in 1...silexPageNumber)
		{
			var silexPage = ps.getPageNumber(i);
			if(silexPage == null) //If page doesn't exist
			{
				silexPage = new org.silex.publication.PageModel();
				ps.pages.push(silexPage);
				ps.recalculatePagesNumbers();
				silexPage.name = options.pageName + silexPage.pageLeftNumber;
				silexPage.deeplink = options.pageName + silexPage.pageLeftNumber;
				silexPage.enabled = true;
			}
		}
		
		//Now going through all PDF pages
		for(pn in pagesLinks.keys())
		{
			if(pn == startPage) //If it's the first page
			{
				templateType = TemplateType.FirstPage;
			} else if (pn == lastPage) //If it's the last page
			{
				templateType = TemplateType.LastPage;
			} else
			{
				templateType = TemplateType.NormalPage;
			}
			var silexPage = ps.getPageNumber(pagesLinks.get(pn));
			if(silexPage == null)
			{
				silexPage = new org.silex.publication.PageModel();
				ps.pages.push(silexPage);
				ps.recalculatePagesNumbers();
				silexPage.name = options.pageName + silexPage.pageLeftNumber; //TODO: Maybe look at if page already exists?
				silexPage.deeplink = options.pageName + silexPage.pageLeftNumber;
			}
			silexPage.enabled = true;			
			//If XML Layer doesn't already exist and template file exist, use it.
			var useTemplate = false;
			if(!php.FileSystem.exists(contentFolder+ options.publicationName + "/" + silexPage.name + ".xml") && php.FileSystem.exists(contentFolder+ options.publicationName + "/" + NORMAL_PAGES))
			{
				switch(templateType)
				{
					case FirstPage:
						php.io.File.copy(contentFolder+ options.publicationName + "/" + FIRST_PAGE, contentFolder+ options.publicationName + "/" + silexPage.name + ".xml");
					case NormalPage:
						php.io.File.copy(contentFolder+ options.publicationName + "/" + NORMAL_PAGES, contentFolder+ options.publicationName + "/" + silexPage.name + ".xml");
					case LastPage:
						php.io.File.copy(contentFolder+ options.publicationName + "/" + LAST_PAGE, contentFolder+ options.publicationName + "/" + silexPage.name + ".xml");
				}
			}
			//Open LayerModel or create it.
			var layer : org.silex.publication.LayerModel;
			try
			{
			//	php.Lib.print("Trying to load from XML " + silexPage.name);
				layer = org.silex.publication.LayerParser.xml2Layer(Xml.parse(File.getContent(contentFolder+ options.publicationName + "/" + silexPage.name + ".xml")));
			//	php.Lib.print("Done loading from XML");
			} catch (e : Dynamic)
			{
			//	php.Lib.print("Exception while loading from XML");
				layer = new org.silex.publication.LayerModel();
			}
			layer.name = silexPage.name;			
			var comp;
			comp = findComponentInLayerByName(layer, if(silexPage.pageLeftNumber == pagesLinks.get(pn)) LEFT_COMPONENT else RIGHT_COMPONENT);

			if(comp != null)
				useTemplate = true;

			if(!useTemplate)
			{
				//Find SubLayer or create it.
				var subLayer = layer.getSubLayerNamed(layerName);
				if(subLayer == null)
				{
					subLayer = new org.silex.publication.SubLayerModel();
					subLayer.id = layerName;
					layer.subLayers.add(subLayer);
				}
				//	trace("Sublayer found or created");
				//trace(subLayer.components);
			
				//Create component.
				comp = new org.silex.publication.ComponentModel();
				comp.properties.set("alpha", 100);
				comp.properties.set("width", Std.int(options.pageWidth));
				comp.properties.set("height", Std.int(options.pageHeight));
				//comp.properties.set(ComponentModel.FIELD_PLAYER_NAME, options.swfDirectoryName + '/' + options.namePrefix + '-' + pn + importExtension);
			
				if(silexPage.pageLeftNumber == pn)
				{
					comp.properties.set("x", Std.int(options.leftPageXOffset));
					comp.properties.set("y", Std.int(options.leftPageYOffset));	
				} else
				{
					comp.properties.set("x", Std.int(options.rightPageXOffset));
					comp.properties.set("y", Std.int(options.rightPageYOffset));
				}
				comp.className = "org.silex.ui.players.Image";
				comp.as2Url = "plugins/baseComponents/as2/org.silex.ui.players.Image.swf";
				comp.html5Url = "html5#org.silex.ui.Image";
			
				comp.properties.set("playerName", options.namePrefix + pn);
				comp.properties.set("rotation", 0);
				comp.properties.set("tabIndex", 1);
				comp.properties.set("tabEnabled", false);
				comp.properties.set("visibleOutOfAdmin", true);
				comp.properties.set("url", options.swfDirectoryName + '/' + options.namePrefix + pn + importExtension);
				comp.properties.set("iconIsDefault", false);
				comp.properties.set("iconIsIcon", false);
				comp.properties.set("iconLayoutName", "minimal.swf");
				comp.properties.set("iconPageName", "new page name");
				comp.properties.set("fadeInStep", 100);
				comp.properties.set("tabChildren", false);
				comp.properties.set("_focusRect", false);
				comp.properties.set("shadowOffsetY", 10);
				comp.properties.set("shadowOffsetX", 10);
				comp.properties.set("shadow", false);
				comp.properties.set("visibleFrame_bool", false);
				comp.properties.set("scaleMode", "none");
				comp.properties.set("useHandCursor", false);
				comp.properties.set("clickable", true);
			
				//trace("Component created");
				//Add it to SubLayer.
				subLayer.components.add(comp);
			} else
			{
				//We are using the template
				comp.properties.set("url", options.swfDirectoryName + '/' + options.namePrefix + pn + importExtension);
			}
			//Save Layer.
			//trace(layer.save("contents/"+ options.publicationName));
//			try
//			{
				layer.save("contents/" + options.publicationName);
//			} catch (e : Dynamic)
//			{
//				return ReturnCode.Error("Couldn't save layer.");
//			}
			//trace("Layer saved");
			//Open preview of this page if it does exist or create it
			var preview;
			if(php.FileSystem.exists(contentFolder + options.publicationName + "/" + silexPage.name + ".jpg"))
			{
				preview = untyped __call__("imagecreatefromjpeg", contentFolder + options.publicationName + "/" + silexPage.name + ".jpg");
			} else
			{
				var layerPrev = org.silex.publication.LayerParser.xml2Layer(Xml.parse(File.getContent(contentFolder+ options.publicationName + "/" + NORMAL_PAGES)));
				var newWidth : Int;
				var newHeight : Int;
				var leftComp = findComponentInLayerByName(layerPrev, LEFT_COMPONENT);
				var rightComp = findComponentInLayerByName(layerPrev, RIGHT_COMPONENT);
				if(ps.hasDoublePages)
				{
					newWidth = Std.int(rightComp.properties.get("x")) + Std.int(rightComp.properties.get("width")) - Std.int(leftComp.properties.get("x"));
					newHeight = Std.int(rightComp.properties.get("y")) + Std.int(rightComp.properties.get("height")) - Std.int(leftComp.properties.get("y"));
					snapLeft = Std.int(leftComp.properties.get("x"));
					snapTop = Std.int(leftComp.properties.get("y"));
				} else
				{
					newWidth = Std.int(rightComp.properties.get("width"));
					newHeight = Std.int(rightComp.properties.get("height"));
					snapLeft = Std.int(rightComp.properties.get("x"));
					snapTop = Std.int(rightComp.properties.get("y"));
				}
				preview = untyped __call__("imagecreatetruecolor", newWidth, newHeight);
			}
			var imgcomponent = untyped __call__("imagecreatefromjpeg", options.swfDirectoryName + "/" + options.namePrefix + pn + ".jpg");
			var size = php.Lib.toHaxeArray(untyped __call__("getimagesize", options.swfDirectoryName + "/" + options.namePrefix + pn + ".jpg"));
			untyped __call__("imagecopyresampled", preview, imgcomponent, Std.parseInt(comp.properties.get("x")) - snapLeft , Std.parseInt(comp.properties.get("y")) - snapTop, 0, 0, comp.properties.get("width"), comp.properties.get("height"), size[0], size[1]);
			untyped __call__("imagejpeg",preview, contentFolder + options.publicationName + "/" + silexPage.name + ".jpg", 75);
			if(!Lambda.has(previewsURL, contentFolder + options.publicationName + "/" + silexPage.name + ".jpg"))
			{
				previewsURL.add(contentFolder + options.publicationName + "/" + silexPage.name + ".jpg");
			}
			//Delete JPG file if it's not included in page as media
			var importExtension = switch(importFormat)
									{
										case "SWF":
											php.FileSystem.deleteFile(options.swfDirectoryName + "/" + options.namePrefix + pn + ".jpg");
										default:
									};
			//trace("preview generated");
		}
			
		//Save the Publication Structure
		try
		{
			File.putContent(contentFolder + options.publicationName + "/structure.xml", org.silex.publication.PublicationStructureParser.ps2XML(ps).toString());
			File.putContent(contentFolder + options.publicationName + "/structure.published.xml", org.silex.publication.PublicationStructureParser.ps2XML(ps).toString());
		} catch(e : Dynamic)
		{
			return ReturnCode.Error("Couldn't save structure.xml and structure.published.xml");
		}
		
		//Construct array for correspondance between pdf page and silex'page'url
		var returnArray = new Array<String>();
		ps.recalculatePagesNumbers();
		for(pn in startPage...lastPage+1)
		{
				returnArray.push("?/"+ options.publicationName + "#/" + ps.getPageNumber(pagesLinks.get(pn)).name);
		}
		return ReturnCode.Done(previewsURL.join("\r"));
	}
	
	private static function createDirPath(path : String)
	{
		var partialPath = "";
		for(s in path.split("/"))
		{
			partialPath += s;
			if(!php.FileSystem.exists(partialPath))
			{
				php.FileSystem.createDirectory(partialPath);
			}
			partialPath += "/";
		}
	}

	private static function concatPath(path : String, command : String)
	{
		if(path != null && path != "")
		{
			if(path.charAt(path.length) != "/")
			{
				return path + "/" + command;
			}
			return path + command;
		}
		return command;
	}
	
	/**
	*  Retrieves a component in a layer by its name
	*/
	private static function findComponentInLayerByName(layer : org.silex.publication.LayerModel, name : String)
	{
		for(sb in layer.subLayers)
		{
			for(comp in sb.components)
			{
				if(comp.properties.get("playerName") == name)
					return comp;
			}
		}
		return null;
	}
	
	private static function debug(msg : String)
	{
		#if SILEX_DEBUG
		var out = php.io.File.append("debug_pdf.txt", false);
		out.writeString(msg + "\r\n");
		out.close();
		#end
	}
}

enum TemplateType
{
	FirstPage;
	NormalPage;
	LastPage;
}

enum ReturnCode
{
	Done(msg : String);
	Error(msg : String);
	PageDoesNotExist;
}
